/*
YUI 3.4.1 (build 4118)
Copyright 2011 Yahoo! Inc. All rights reserved.
Licensed under the BSD License.
http://yuilibrary.com/license/
*/
YUI.add('graphics-svg', function(Y) {

var SHAPE = "svgShape",
        Y_LANG = Y.Lang,
        AttributeLite = Y.AttributeLite,
        SVGGraphic,
    SVGShape,
        SVGCircle,
        SVGRect,
        SVGPath,
        SVGEllipse,
    SVGPieSlice,
    DOCUMENT = Y.config.doc;

function SVGDrawing(){}

/**
 * <a href="http://www.w3.org/TR/SVG/">SVG</a> implementation of the <a href="Drawing.html">`Drawing`</a> class. 
 * `SVGDrawing` is not intended to be used directly. Instead, use the <a href="Drawing.html">`Drawing`</a> class. 
 * If the browser has <a href="http://www.w3.org/TR/SVG/">SVG</a> capabilities, the <a href="Drawing.html">`Drawing`</a> 
 * class will point to the `SVGDrawing` class.
 *
 * @module graphics
 * @class SVGDrawing
 * @constructor
 */
SVGDrawing.prototype = {
    /**
     * Indicates the type of shape
     *
     * @private
     * @property _type
     * @readOnly
     * @type String
     */
    _type: "path",
   
    /**
     * Draws a bezier curve.
     *
     * @method curveTo
     * @param {Number} cp1x x-coordinate for the first control point.
     * @param {Number} cp1y y-coordinate for the first control point.
     * @param {Number} cp2x x-coordinate for the second control point.
     * @param {Number} cp2y y-coordinate for the second control point.
     * @param {Number} x x-coordinate for the end point.
     * @param {Number} y y-coordinate for the end point.
     */
    curveTo: function(cp1x, cp1y, cp2x, cp2y, x, y) {
        var pathArrayLen,
            currentArray,
            hiX,
            loX,
            hiY,
            loY;
        if(this._pathType !== "C")
        {
            this._pathType = "C";
            currentArray = ["C"];
            this._pathArray.push(currentArray);
        }
        else
        {
            currentArray = this._pathArray[Math.max(0, this._pathArray.length - 1)];
            if(!currentArray)
            {
                currentArray = [];
                this._pathArray.push(currentArray);
            }
        }
        pathArrayLen = this._pathArray.length - 1;
        this._pathArray[pathArrayLen] = this._pathArray[pathArrayLen].concat([Math.round(cp1x), Math.round(cp1y), Math.round(cp2x) , Math.round(cp2y), x, y]);
        hiX = Math.max(x, Math.max(cp1x, cp2x));
        hiY = Math.max(y, Math.max(cp1y, cp2y));
        loX = Math.min(x, Math.min(cp1x, cp2x));
        loY = Math.min(y, Math.min(cp1y, cp2y));
        this._trackSize(hiX, hiY);
        this._trackSize(loX, loY);
    },

    /**
     * Draws a quadratic bezier curve.
     *
     * @method quadraticCurveTo
     * @param {Number} cpx x-coordinate for the control point.
     * @param {Number} cpy y-coordinate for the control point.
     * @param {Number} x x-coordinate for the end point.
     * @param {Number} y y-coordinate for the end point.
     */
    quadraticCurveTo: function(cpx, cpy, x, y) {
        var pathArrayLen,
            currentArray,
            hiX,
            loX,
            hiY,
            loY;
        if(this._pathType !== "Q")
        {
            this._pathType = "Q";
            currentArray = ["Q"];
            this._pathArray.push(currentArray);
        }
        else
        {
            currentArray = this._pathArray[Math.max(0, this._pathArray.length - 1)];
            if(!currentArray)
            {
                currentArray = [];
                this._pathArray.push(currentArray);
            }
        }
        pathArrayLen = this._pathArray.length - 1;
        this._pathArray[pathArrayLen] = this._pathArray[pathArrayLen].concat([Math.round(cpx), Math.round(cpy), Math.round(x), Math.round(y)]);
        hiX = Math.max(x, cpx);
        hiY = Math.max(y, cpy);
        loX = Math.min(x, cpx);
        loY = Math.min(y, cpy);
        this._trackSize(hiX, hiY);
        this._trackSize(loX, loY);
    },

    /**
     * Draws a rectangle.
     *
     * @method drawRect
     * @param {Number} x x-coordinate
     * @param {Number} y y-coordinate
     * @param {Number} w width
     * @param {Number} h height
     */
    drawRect: function(x, y, w, h) {
        this.moveTo(x, y);
        this.lineTo(x + w, y);
        this.lineTo(x + w, y + h);
        this.lineTo(x, y + h);
        this.lineTo(x, y);
    },

    /**
     * Draws a rectangle with rounded corners.
     * 
     * @method drawRect
     * @param {Number} x x-coordinate
     * @param {Number} y y-coordinate
     * @param {Number} w width
     * @param {Number} h height
     * @param {Number} ew width of the ellipse used to draw the rounded corners
     * @param {Number} eh height of the ellipse used to draw the rounded corners
     */
    drawRoundRect: function(x, y, w, h, ew, eh) {
        this.moveTo(x, y + eh);
        this.lineTo(x, y + h - eh);
        this.quadraticCurveTo(x, y + h, x + ew, y + h);
        this.lineTo(x + w - ew, y + h);
        this.quadraticCurveTo(x + w, y + h, x + w, y + h - eh);
        this.lineTo(x + w, y + eh);
        this.quadraticCurveTo(x + w, y, x + w - ew, y);
        this.lineTo(x + ew, y);
        this.quadraticCurveTo(x, y, x, y + eh);
        },

    /**
     * Draws a wedge.
     *
     * @method drawWedge
     * @param {Number} x x-coordinate of the wedge's center point
     * @param {Number} y y-coordinate of the wedge's center point
     * @param {Number} startAngle starting angle in degrees
     * @param {Number} arc sweep of the wedge. Negative values draw clockwise.
     * @param {Number} radius radius of wedge. If [optional] yRadius is defined, then radius is the x radius.
     * @param {Number} yRadius [optional] y radius for wedge.
     * @private
     */
    drawWedge: function(x, y, startAngle, arc, radius, yRadius)
    {
        var segs,
            segAngle,
            theta,
            angle,
            angleMid,
            ax,
            ay,
            bx,
            by,
            cx,
            cy,
            i = 0,
            diameter = radius * 2,
            currentArray,
            pathArrayLen;
        yRadius = yRadius || radius;
        if(this._pathType != "M")
        {
            this._pathType = "M";
            currentArray = ["M"];
            this._pathArray.push(currentArray);
        }
        else
        {
            currentArray = this._getCurrentArray(); 
        }
        pathArrayLen = this._pathArray.length - 1;
        this._pathArray[pathArrayLen].push(x); 
        this._pathArray[pathArrayLen].push(x); 
        
        // limit sweep to reasonable numbers
        if(Math.abs(arc) > 360)
        {
            arc = 360;
        }
        
        // First we calculate how many segments are needed
        // for a smooth arc.
        segs = Math.ceil(Math.abs(arc) / 45);
        
        // Now calculate the sweep of each segment.
        segAngle = arc / segs;
        
        // The math requires radians rather than degrees. To convert from degrees
        // use the formula (degrees/180)*Math.PI to get radians.
        theta = -(segAngle / 180) * Math.PI;
        
        // convert angle startAngle to radians
        angle = (startAngle / 180) * Math.PI;
        if(segs > 0)
        {
            // draw a line from the center to the start of the curve
            ax = x + Math.cos(startAngle / 180 * Math.PI) * radius;
            ay = y + Math.sin(startAngle / 180 * Math.PI) * yRadius;
            this._pathType = "L";
            pathArrayLen++;
            this._pathArray[pathArrayLen] = ["L"];
            this._pathArray[pathArrayLen].push(Math.round(ax));
            this._pathArray[pathArrayLen].push(Math.round(ay));
            pathArrayLen++; 
            this._pathType = "Q";
            this._pathArray[pathArrayLen] = ["Q"];
            for(; i < segs; ++i)
            {
                angle += theta;
                angleMid = angle - (theta / 2);
                bx = x + Math.cos(angle) * radius;
                by = y + Math.sin(angle) * yRadius;
                cx = x + Math.cos(angleMid) * (radius / Math.cos(theta / 2));
                cy = y + Math.sin(angleMid) * (yRadius / Math.cos(theta / 2));
                this._pathArray[pathArrayLen].push(Math.round(cx));
                this._pathArray[pathArrayLen].push(Math.round(cy));
                this._pathArray[pathArrayLen].push(Math.round(bx));
                this._pathArray[pathArrayLen].push(Math.round(by));
            }
        }
        this._trackSize(diameter, diameter); 
        return this;
    },
    
    /**
     * Draws a line segment using the current line style from the current drawing position to the specified x and y coordinates.
     * 
     * @method lineTo
     * @param {Number} point1 x-coordinate for the end point.
     * @param {Number} point2 y-coordinate for the end point.
     */
    lineTo: function(point1, point2, etc) {
        var args = arguments,
            i,
            len,
            pathArrayLen,
            currentArray;
        this._pathArray = this._pathArray || [];
        if (typeof point1 === 'string' || typeof point1 === 'number') {
            args = [[point1, point2]];
        }
        len = args.length;
        this._shapeType = "path";
        if(this._pathType !== "L")
        {
            this._pathType = "L";
            currentArray = ['L'];
            this._pathArray.push(currentArray);
        }
        else
        {
            currentArray = this._getCurrentArray();
        }
        pathArrayLen = this._pathArray.length - 1;
        for (i = 0; i < len; ++i) {
            this._pathArray[pathArrayLen].push(args[i][0]);
            this._pathArray[pathArrayLen].push(args[i][1]);
            this._trackSize.apply(this, args[i]);
        }
    },

    _getCurrentArray: function()
    {
        var currentArray = this._pathArray[Math.max(0, this._pathArray.length - 1)];
        if(!currentArray)
        {
            currentArray = [];
            this._pathArray.push(currentArray);
        }
        return currentArray;
    },

    /**
     * Moves the current drawing position to specified x and y coordinates.
     *
     * @method moveTo
     * @param {Number} x x-coordinate for the end point.
     * @param {Number} y y-coordinate for the end point.
     */
    moveTo: function(x, y) {
        var pathArrayLen,
            currentArray;
        this._pathArray = this._pathArray || [];
        if(this._pathType != "M")
        {
            this._pathType = "M";
            currentArray = ["M"];
            this._pathArray.push(currentArray);
        }
        else
        {
            currentArray = this._getCurrentArray(); 
        }
        pathArrayLen = this._pathArray.length - 1;
        this._pathArray[pathArrayLen] = this._pathArray[pathArrayLen].concat([x, y]);
        this._trackSize(x, y);
    },
 
    /**
     * Completes a drawing operation. 
     *
     * @method end
     */
    end: function()
    {
        this._closePath();
        this._graphic.addToRedrawQueue(this);    
    },

    /**
     * Clears the path.
     *
     * @method clear
     */
    clear: function()
    {
        this._left = 0;
        this._right = 0;
        this._top = 0;
        this._bottom = 0;
        this._pathArray = [];
        this._path = "";
    },

    /**
     * Draws the path.
     *
     * @method _closePath
     * @private
     */
    _closePath: function()
    {
        var pathArray,
            segmentArray,
            pathType,
            len,
            val,
            val2,
            i,
            path = "",
            node = this.node,
            left = this._left,
            top = this._top,
            fill = this.get("fill");
        if(this._pathArray)
        {
            pathArray = this._pathArray.concat();
            while(pathArray && pathArray.length > 0)
            {
                segmentArray = pathArray.shift();
                len = segmentArray.length;
                pathType = segmentArray[0];
                path += " " + pathType + (segmentArray[1] - left);
                switch(pathType)
                {
                    case "L" :
                    case "M" :
                    case "Q" :
                        for(i = 2; i < len; ++i)
                        {
                            val = (i % 2 === 0) ? top : left;
                            val = segmentArray[i] - val;
                            path += ", " + val;
                        }
                    break;
                    case "C" :
                        for(i = 2; i < len; ++i)
                        {
                            val = (i % 2 === 0) ? top : left;
                            val2 = segmentArray[i];
                            val2 -= val;
                            path += " " + val2;
                        }
                    break;

                }
            }
            if(fill && fill.color)
            {
                path += 'z';
            }
            if(path)
            {
                node.setAttribute("d", path);
            }
            
            this._path = path;
            this._fillChangeHandler();
            this._strokeChangeHandler();
            this._updateTransform();
        }
    },

    /**
     * Updates the size of the graphics object
     *
     * @method _trackSize
     * @param {Number} w width
     * @param {Number} h height
     * @private
     */
    _trackSize: function(w, h) {
        if (w > this._right) {
            this._right = w;
        }
        if(w < this._left)
        {
            this._left = w;    
        }
        if (h < this._top)
        {
            this._top = h;
        }
        if (h > this._bottom) 
        {
            this._bottom = h;
        }
        this._width = this._right - this._left;
        this._height = this._bottom - this._top;
    }
};
Y.SVGDrawing = SVGDrawing;
/**
 * <a href="http://www.w3.org/TR/SVG/">SVG</a> implementation of the <a href="Shape.html">`Shape`</a> class. 
 * `SVGShape` is not intended to be used directly. Instead, use the <a href="Shape.html">`Shape`</a> class. 
 * If the browser has <a href="http://www.w3.org/TR/SVG/">SVG</a> capabilities, the <a href="Shape.html">`Shape`</a> 
 * class will point to the `SVGShape` class.
 *
 * @module graphics
 * @class SVGShape
 * @constructor
 * @param {Object} cfg (optional) Attribute configs
 */
SVGShape = function(cfg)
{
    this._transforms = [];
    this.matrix = new Y.Matrix();
    SVGShape.superclass.constructor.apply(this, arguments);
};

SVGShape.NAME = "svgShape";

Y.extend(SVGShape, Y.BaseGraphic, Y.mix({
    /**
     * Init method, invoked during construction.
     * Calls `initializer` method.
     *
     * @method init
     * @protected
     */
        init: function()
        {
                this.initializer.apply(this, arguments);
        },

        /**
         * Initializes the shape
         *
         * @private
         * @method initializer
         */
        initializer: function(cfg)
        {
                var host = this;
                host.createNode(); 
                host._graphic = cfg.graphic;
                host._updateHandler();
        },
   
        /**
         * Add a class name to each node.
         *
         * @method addClass
         * @param {String} className the class name to add to the node's class attribute 
         */
        addClass: function(className)
        {
                var node = this.node;
                node.className.baseVal = Y_LANG.trim([node.className.baseVal, className].join(' '));
        },

        /**
         * Removes a class name from each node.
         *
         * @method removeClass
         * @param {String} className the class name to remove from the node's class attribute
         */
        removeClass: function(className)
        {
                var node = this.node,
                        classString = node.className.baseVal;
                classString = classString.replace(new RegExp(className + ' '), className).replace(new RegExp(className), '');
                node.className.baseVal = classString;
        },

        /**
         * Gets the current position of the node in page coordinates.
         *
         * @method getXY
         * @return Array The XY position of the shape.
         */
        getXY: function()
        {
                var graphic = this._graphic,
                        parentXY = graphic.getXY(),
                        x = this.get("x"),
                        y = this.get("y");
                return [parentXY[0] + x, parentXY[1] + y];
        },

        /**
         * Set the position of the shape in page coordinates, regardless of how the node is positioned.
         *
         * @method setXY
         * @param {Array} Contains x & y values for new position (coordinates are page-based)
         */
        setXY: function(xy)
        {
                var graphic = this._graphic,
                        parentXY = graphic.getXY();
                this.set("x", xy[0] - parentXY[0]);
                this.set("y", xy[1] - parentXY[1]);
        },

        /**
         * Determines whether the node is an ancestor of another HTML element in the DOM hierarchy. 
         *
         * @method contains
         * @param {SVGShape | HTMLElement} needle The possible node or descendent
         * @return Boolean Whether or not this shape is the needle or its ancestor.
         */
        contains: function(needle)
        {
                return needle === Y.one(this.node);
        },

        /**
         * Compares nodes to determine if they match.
         * Node instances can be compared to each other and/or HTMLElements.
         * @method compareTo
         * @param {HTMLElement | Node} refNode The reference node to compare to the node.
         * @return {Boolean} True if the nodes match, false if they do not.
         */
        compareTo: function(refNode) {
                var node = this.node;

                return node === refNode;
        },

        /**
         * Test if the supplied node matches the supplied selector.
         *
         * @method test
         * @param {String} selector The CSS selector to test against.
         * @return Boolean Wheter or not the shape matches the selector.
         */
        test: function(selector)
        {
                return Y.Selector.test(this.node, selector);
        },
        
        /**
         * Value function for fill attribute
         *
         * @private
         * @method _getDefaultFill
         * @return Object
         */
        _getDefaultFill: function() {
                return {
                        type: "solid",
                        opacity: 1,
                        cx: 0.5,
                        cy: 0.5,
                        fx: 0.5,
                        fy: 0.5,
                        r: 0.5
                };
        },
        
        /**
         * Value function for stroke attribute
         *
         * @private
         * @method _getDefaultStroke
         * @return Object
         */
        _getDefaultStroke: function() 
        {
                return {
                        weight: 1,
                        dashstyle: "none",
                        color: "#000",
                        opacity: 1.0
                };
        },

        /**
         * Creates the dom node for the shape.
         *
     * @method createNode
         * @return HTMLElement
         * @private
         */
        createNode: function()
        {
                var node = DOCUMENT.createElementNS("http://www.w3.org/2000/svg", "svg:" + this._type),
                        id = this.get("id"),
                        pointerEvents = this.get("pointerEvents");
                this.node = node;
                this.addClass("yui3-" + SHAPE + " yui3-" + this.name);
                if(id)
                {
                        node.setAttribute("id", id);
                }
                if(pointerEvents)
                {
                        node.setAttribute("pointer-events", pointerEvents);
                }
        },
        

        /**
     * Overrides default `on` method. Checks to see if its a dom interaction event. If so, 
     * return an event attached to the `node` element. If not, return the normal functionality.
     *
     * @method on
     * @param {String} type event type
     * @param {Object} callback function
         * @private
         */
        on: function(type, fn)
        {
                if(Y.Node.DOM_EVENTS[type])
                {
                        return Y.one("#" +  this.get("id")).on(type, fn);
                }
                return Y.on.apply(this, arguments);
        },

        /**
         * Adds a stroke to the shape node.
         *
         * @method _strokeChangeHandler
         * @private
         */
        _strokeChangeHandler: function(e)
        {
                var node = this.node,
                        stroke = this.get("stroke"),
                        strokeOpacity,
                        dashstyle,
                        dash,
                        linejoin;
                if(stroke && stroke.weight && stroke.weight > 0)
                {
                        linejoin = stroke.linejoin || "round";
                        strokeOpacity = parseFloat(stroke.opacity);
                        dashstyle = stroke.dashstyle || "none";
                        dash = Y_LANG.isArray(dashstyle) ? dashstyle.toString() : dashstyle;
                        stroke.color = stroke.color || "#000000";
                        stroke.weight = stroke.weight || 1;
                        stroke.opacity = Y_LANG.isNumber(strokeOpacity) ? strokeOpacity : 1;
                        stroke.linecap = stroke.linecap || "butt";
                        node.setAttribute("stroke-dasharray", dash);
                        node.setAttribute("stroke", stroke.color);
                        node.setAttribute("stroke-linecap", stroke.linecap);
                        node.setAttribute("stroke-width",  stroke.weight);
                        node.setAttribute("stroke-opacity", stroke.opacity);
                        if(linejoin == "round" || linejoin == "bevel")
                        {
                                node.setAttribute("stroke-linejoin", linejoin);
                        }
                        else
                        {
                                linejoin = parseInt(linejoin, 10);
                                if(Y_LANG.isNumber(linejoin))
                                {
                                        node.setAttribute("stroke-miterlimit",  Math.max(linejoin, 1));
                                        node.setAttribute("stroke-linejoin", "miter");
                                }
                        }
                }
                else
                {
                        node.setAttribute("stroke", "none");
                }
        },
        
        /**
         * Adds a fill to the shape node.
         *
         * @method _fillChangeHandler
         * @private
         */
        _fillChangeHandler: function(e)
        {
                var node = this.node,
                        fill = this.get("fill"),
                        fillOpacity,
                        type;
                if(fill)
                {
                        type = fill.type;
                        if(type == "linear" || type == "radial")
                        {
                                this._setGradientFill(fill);
                                node.setAttribute("fill", "url(#grad" + this.get("id") + ")");
                        }
                        else if(!fill.color)
                        {
                                node.setAttribute("fill", "none");
                        }
                        else
                        {
                fillOpacity = parseFloat(fill.opacity);
                                fillOpacity = Y_LANG.isNumber(fillOpacity) ? fillOpacity : 1;
                                node.setAttribute("fill", fill.color);
                                node.setAttribute("fill-opacity", fillOpacity);
                        }
                }
                else
                {
                        node.setAttribute("fill", "none");
                }
        },

        /**
         * Creates a gradient fill
         *
         * @method _setGradientFill
         * @param {String} type gradient type
         * @private
         */
        _setGradientFill: function(fill) {
                var offset,
                        opacity,
                        color,
                        stopNode,
                        isNumber = Y_LANG.isNumber,
                        graphic = this._graphic,
                        type = fill.type, 
                        gradientNode = graphic.getGradientNode("grad" + this.get("id"), type),
                        stops = fill.stops,
                        w = this.get("width"),
                        h = this.get("height"),
                        rotation = fill.rotation,
                        radCon = Math.PI/180,
            tanRadians = parseFloat(parseFloat(Math.tan(rotation * radCon)).toFixed(8)),
            i,
                        len,
                        def,
                        stop,
                        x1 = "0%", 
                        x2 = "100%", 
                        y1 = "0%", 
                        y2 = "0%",
                        cx = fill.cx,
                        cy = fill.cy,
                        fx = fill.fx,
                        fy = fill.fy,
                        r = fill.r;
                if(type == "linear")
                {
            cx = w/2;
            cy = h/2;
            if(Math.abs(tanRadians) * w/2 >= h/2)
            {
                if(rotation < 180)
                {
                    y1 = 0;
                    y2 = h;
                }
                else
                {
                    y1 = h;
                    y2 = 0;
                }
                x1 = cx - ((cy - y1)/tanRadians);
                x2 = cx - ((cy - y2)/tanRadians); 
            }
            else
            {
                if(rotation > 90 && rotation < 270)
                {
                    x1 = w;
                    x2 = 0;
                }
                else
                {
                    x1 = 0;
                    x2 = w;
                }
                y1 = ((tanRadians * (cx - x1)) - cy) * -1;
                y2 = ((tanRadians * (cx - x2)) - cy) * -1;
            }
            gradientNode.setAttribute("spreadMethod", "pad");
                        gradientNode.setAttribute("width", w);
                        gradientNode.setAttribute("height", h);
            gradientNode.setAttribute("x1", Math.round(100 * x1/w) + "%");
            gradientNode.setAttribute("y1", Math.round(100 * y1/h) + "%");
            gradientNode.setAttribute("x2", Math.round(100 * x2/w) + "%");
            gradientNode.setAttribute("y2", Math.round(100 * y2/h) + "%");
                }
                else
                {
                        gradientNode.setAttribute("cx", (cx * 100) + "%");
                        gradientNode.setAttribute("cy", (cy * 100) + "%");
                        gradientNode.setAttribute("fx", (fx * 100) + "%");
                        gradientNode.setAttribute("fy", (fy * 100) + "%");
                        gradientNode.setAttribute("r", (r * 100) + "%");
                }
                
                len = stops.length;
                def = 0;
                for(i = 0; i < len; ++i)
                {
                        stop = stops[i];
                        opacity = stop.opacity;
                        color = stop.color;
                        offset = stop.offset || i/(len - 1);
                        offset = Math.round(offset * 100) + "%";
                        opacity = isNumber(opacity) ? opacity : 1;
                        opacity = Math.max(0, Math.min(1, opacity));
                        def = (i + 1) / len;
                        stopNode = graphic._createGraphicNode("stop");
                        stopNode.setAttribute("offset", offset);
                        stopNode.setAttribute("stop-color", color);
                        stopNode.setAttribute("stop-opacity", opacity);
                        gradientNode.appendChild(stopNode);
                }
        },

    /**
     * Sets the value of an attribute.
     *
     * @method set
     * @param {String|Object} name The name of the attribute. Alternatively, an object of key value pairs can 
     * be passed in to set multiple attributes at once.
     * @param {Any} value The value to set the attribute to. This value is ignored if an object is received as 
     * the name param.
     */
        set: function() 
        {
                var host = this;
                AttributeLite.prototype.set.apply(host, arguments);
                if(host.initialized)
                {
                        host._updateHandler();
                }
        },

        /**
         * Specifies a 2d translation.
         *
         * @method translate
         * @param {Number} x The value to transate on the x-axis.
         * @param {Number} y The value to translate on the y-axis.
         */
        translate: function(x, y)
        {
                this._translateX += x;
                this._translateY += y;
                this._addTransform("translate", arguments);
        },

        /**
         * Translates the shape along the x-axis. When translating x and y coordinates,
         * use the `translate` method.
         *
         * @method translateX
         * @param {Number} x The value to translate.
         */
        translateX: function(x)
    {
        this._translateX += x;
        this._addTransform("translateX", arguments);
    },

        /**
         * Translates the shape along the y-axis. When translating x and y coordinates,
         * use the `translate` method.
         *
         * @method translateY
         * @param {Number} y The value to translate.
         */
        translateY: function(y)
    {
        this._translateY += y;
        this._addTransform("translateY", arguments);
    },

    /**
     * Skews the shape around the x-axis and y-axis.
     *
     * @method skew
     * @param {Number} x The value to skew on the x-axis.
     * @param {Number} y The value to skew on the y-axis.
     */
    skew: function(x, y)
    {
        this._addTransform("skew", arguments);
    },

        /**
         * Skews the shape around the x-axis.
         *
         * @method skewX
         * @param {Number} x x-coordinate
         */
         skewX: function(x)
         {
                this._addTransform("skewX", arguments);
         },

        /**
         * Skews the shape around the y-axis.
         *
         * @method skewY
         * @param {Number} y y-coordinate
         */
         skewY: function(y)
         {
                this._addTransform("skewY", arguments);
         },

        /**
     * Storage for `rotation` atribute.
     *
     * @property _rotation
     * @type Number
         * @private
         */
         _rotation: 0,

        /**
         * Rotates the shape clockwise around it transformOrigin.
         *
         * @method rotate
         * @param {Number} deg The degree of the rotation.
         */
         rotate: function(deg)
         {
                this._rotation = deg;
                this._addTransform("rotate", arguments);
         },

        /**
         * Specifies a 2d scaling operation.
         *
         * @method scale
         * @param {Number} val
         */
        scale: function(x, y)
        {
                this._addTransform("scale", arguments);
        },

    /**
     * Adds a transform to the shape.
     *
     * @method _addTransform
     * @param {String} type The transform being applied.
     * @param {Array} args The arguments for the transform.
         * @private
         */
        _addTransform: function(type, args)
        {
        args = Y.Array(args);
        this._transform = Y_LANG.trim(this._transform + " " + type + "(" + args.join(", ") + ")");
        args.unshift(type);
        this._transforms.push(args);
        if(this.initialized)
        {
            this._updateTransform();
        }
        },

        /**
     * Applies all transforms.
     *
     * @method _updateTransform
         * @private
         */
        _updateTransform: function()
        {
                var isPath = this._type == "path",
                    node = this.node,
                        key,
                        transform,
                        transformOrigin,
                        x,
                        y,
            tx,
            ty,
            matrix = this.matrix,
            i = 0,
            len = this._transforms.length;

        if(isPath || (this._transforms && this._transforms.length > 0))
                {
            x = this.get("x");
            y = this.get("y");
            
            if(isPath)
            {
                x += this._left;
                y += this._top;
                matrix.init({dx: x, dy: y});
                x = 0;
                y = 0;
            }
            for(; i < len; ++i)
            {
                key = this._transforms[i].shift();
                if(key)
                {
                    if(key == "rotate" || key == "scale")
                    {
                                        transformOrigin = this.get("transformOrigin");
                        tx = x + (transformOrigin[0] * this.get("width"));
                        ty = y + (transformOrigin[1] * this.get("height")); 
                        matrix.translate(tx, ty);
                        matrix[key].apply(matrix, this._transforms[i]); 
                        matrix.translate(0 - tx, 0 - ty);
                    }
                    else
                    {
                        matrix[key].apply(matrix, this._transforms[i]); 
                    }
                }
                if(isPath)
                {
                    this._transforms[i].unshift(key);
                }
                        }
            transform = "matrix(" + matrix.a + "," + 
                            matrix.b + "," + 
                            matrix.c + "," + 
                            matrix.d + "," + 
                            matrix.dx + "," +
                            matrix.dy + ")";
                }
        this._graphic.addToRedrawQueue(this);    
        if(transform)
                {
            node.setAttribute("transform", transform);
        }
        if(!isPath)
        {
            this._transforms = [];
        }
        },

        /**
         * Draws the shape.
         *
         * @method _draw
         * @private
         */
        _draw: function()
        {
                var node = this.node;
                node.setAttribute("width", this.get("width"));
                node.setAttribute("height", this.get("height"));
                node.setAttribute("x", this.get("x"));
                node.setAttribute("y", this.get("y"));
                node.style.left = this.get("x") + "px";
                node.style.top = this.get("y") + "px";
                this._fillChangeHandler();
                this._strokeChangeHandler();
                this._updateTransform();
        },

        /**
     * Updates `Shape` based on attribute changes.
     *
     * @method _updateHandler
         * @private
         */
        _updateHandler: function(e)
        {
                this._draw();
        },
        
        /**
         * Storage for translateX
         *
     * @property _translateX
     * @type Number
         * @private
         */
        _translateX: 0,

        /**
         * Storage for translateY
         *
     * @property _translateY
     * @type Number
         * @private
         */
        _translateY: 0,
    
    /**
     * Storage for the transform attribute.
     *
     * @property _transform
     * @type String
     * @private
     */
    _transform: "",

        /**
         * Returns the bounds for a shape.
         *
     * Calculates the a new bounding box from the original corner coordinates (base on size and position) and the transform matrix.
     * The calculated bounding box is used by the graphic instance to calculate its viewBox. 
     *
         * @method getBounds
         * @return Object
         */
        getBounds: function()
        {
            var type = this._type,
            wt,
            bounds = {},
            matrix = this.matrix,
            a = matrix.a,
            b = matrix.b,
            c = matrix.c,
            d = matrix.d,
            dx = matrix.dx,
            dy = matrix.dy,
            w = this.get("width"),
            h = this.get("height"),
            //The svg path element does not have x and y coordinates. Shapes based on path use translate to "fake" x and y. As a
            //result, these values will show up in the transform matrix and should not be used in any conversion formula.
            left = type == "path" ? 0 : this.get("x"), 
            top = type == "path" ? 0 : this.get("y"), 
            right = left + w,
            bottom = top + h,
                        stroke = this.get("stroke"),
            //[x1, y1]
            x1 = (a * left + c * top + dx), 
            y1 = (b * left + d * top + dy),
            //[x2, y2]
            x2 = (a * right + c * top + dx),
            y2 = (b * right + d * top + dy),
            //[x3, y3]
            x3 = (a * left + c * bottom + dx),
            y3 = (b * left + d * bottom + dy),
            //[x4, y4]
            x4 = (a * right + c * bottom + dx),
            y4 = (b * right + d * bottom + dy);
        bounds.left = Math.min(x3, Math.min(x1, Math.min(x2, x4)));
        bounds.right = Math.max(x3, Math.max(x1, Math.max(x2, x4)));
        bounds.top = Math.min(y2, Math.min(y4, Math.min(y3, y1)));
        bounds.bottom = Math.max(y2, Math.max(y4, Math.max(y3, y1)));
        //if there is a stroke, extend the bounds to accomodate
        if(stroke && stroke.weight)
                {
                        wt = stroke.weight;
            bounds.left -= wt;
            bounds.right += wt;
            bounds.top -= wt;
            bounds.bottom += wt;
                }
        return bounds;
        },

    /**
     * Returns the x coordinate for a bounding box's corner based on the corner's original x/y coordinates, rotation and transform origin of the rotation.
     *
     * @method _getRotatedCornerX
     * @param {Number} x original x-coordinate of corner
     * @param {Number} y original y-coordinate of corner
     * @param {Number} tox transform origin x-coordinate of rotation
     * @param {Number} toy transform origin y-coordinate of rotation
     * @param {Number} cosRadians cosine (in radians) of rotation
     * @param {Number} sinRadians sin (in radians) or rotation
     * @return Number
     * @private
     */
    _getRotatedCornerX: function(x, y, tox, toy, cosRadians, sinRadians)
    {
        return (tox + (x - tox) * cosRadians + (y - toy) * sinRadians);
    },

    /**
     * Returns the y coordinate for a bounding box's corner based on the corner's original x/y coordinates, rotation and transform origin of the rotation.
     *
     * @method _getRotatedCornerY
     * @param {Number} x original x-coordinate of corner
     * @param {Number} y original y-coordinate of corner
     * @param {Number} tox transform origin x-coordinate of rotation
     * @param {Number} toy transform origin y-coordinate of rotation
     * @param {Number} cosRadians cosine (in radians) of rotation
     * @param {Number} sinRadians sin (in radians) or rotation
     * @return Number
     * @private
     */
    _getRotatedCornerY: function(x, y, tox, toy, cosRadians, sinRadians)
    {
        return (toy - (x - tox) * sinRadians + (y - toy) * cosRadians);
    },

    /**
     * Destroys the instance.
     *
     * @method destroy
     */
    destroy: function()
    {
        if(this._graphic && this._graphic._contentNode)
        {
            this._graphic._contentNode.removeChild(this.node);
        }
    }
 }, Y.SVGDrawing.prototype));
        
SVGShape.ATTRS = {
        /**
         * An array of x, y values which indicates the transformOrigin in which to rotate the shape. Valid values range between 0 and 1 representing a 
         * fraction of the shape's corresponding bounding box dimension. The default value is [0.5, 0.5].
         *
         * @config transformOrigin
         * @type Array
         */
        transformOrigin: {
                valueFn: function()
                {
                        return [0.5, 0.5];
                }
        },
        
    /**
     * <p>A string containing, in order, transform operations applied to the shape instance. The `transform` string can contain the following values:
     *     
     *    <dl>
     *        <dt>rotate</dt><dd>Rotates the shape clockwise around it transformOrigin.</dd>
     *        <dt>translate</dt><dd>Specifies a 2d translation.</dd>
     *        <dt>skew</dt><dd>Skews the shape around the x-axis and y-axis.</dd>
     *        <dt>scale</dt><dd>Specifies a 2d scaling operation.</dd>
     *        <dt>translateX</dt><dd>Translates the shape along the x-axis.</dd>
     *        <dt>translateY</dt><dd>Translates the shape along the y-axis.</dd>
     *        <dt>skewX</dt><dd>Skews the shape around the x-axis.</dd>
     *        <dt>skewY</dt><dd>Skews the shape around the y-axis.</dd>
     *    </dl>
     * </p>
     * <p>Applying transforms through the transform attribute will reset the transform matrix and apply a new transform. The shape class also contains corresponding methods for each transform
     * that will apply the transform to the current matrix. The below code illustrates how you might use the `transform` attribute to instantiate a recangle with a rotation of 45 degrees.</p>
            var myRect = new Y.Rect({
                type:"rect",
                width: 50,
                height: 40,
                transform: "rotate(45)"
            };
     * <p>The code below would apply `translate` and `rotate` to an existing shape.</p>
    
        myRect.set("transform", "translate(40, 50) rotate(45)");
         * @config transform
     * @type String  
         */
        transform: {
                setter: function(val)
                {
            this.matrix.init(); 
                    this._transforms = this.matrix.getTransformArray(val);
            this._transform = val;
            if(this.initialized)
            {
                this._updateTransform();
            }
            return val;
                },

        getter: function()
        {
            return this._transform;
        }
        },

        /**
         * Unique id for class instance.
         *
         * @config id
         * @type String
         */
        id: {
                valueFn: function()
                {
                        return Y.guid();
                },

                setter: function(val)
                {
                        var node = this.node;
                        if(node)
                        {
                                node.setAttribute("id", val);
                        }
                        return val;
                }
        },

        /**
         * Indicates the x position of shape.
         *
         * @config x
         * @type Number
         */
        x: {
                value: 0
        },

        /**
         * Indicates the y position of shape.
         *
         * @config y
         * @type Number
         */
        y: {
                value: 0
        },

        /**
         * Indicates the width of the shape
         *
         * @config width
         * @type Number
         */
        width: {
        value: 0
    },

        /**
         * Indicates the height of the shape
         * 
         * @config height
         * @type Number
         */
        height: {
        value: 0
    },

        /**
         * Indicates whether the shape is visible.
         *
         * @config visible
         * @type Boolean
         */
        visible: {
                value: true,

                setter: function(val){
                        var visibility = val ? "visible" : "hidden";
                        this.node.style.visibility = visibility;
                        return val;
                }
        },

        /**
         * Contains information about the fill of the shape. 
     *  <dl>
     *      <dt>color</dt><dd>The color of the fill.</dd>
     *      <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the fill. The default value is 1.</dd>
     *      <dt>type</dt><dd>Type of fill.
     *          <dl>
     *              <dt>solid</dt><dd>Solid single color fill. (default)</dd>
     *              <dt>linear</dt><dd>Linear gradient fill.</dd>
     *              <dt>radial</dt><dd>Radial gradient fill.</dd>
     *          </dl>
     *      </dd>
     *  </dl>
     *  <p>If a `linear` or `radial` is specified as the fill type. The following additional property is used:
     *  <dl>
     *      <dt>stops</dt><dd>An array of objects containing the following properties:
     *          <dl>
     *              <dt>color</dt><dd>The color of the stop.</dd>
     *              <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stop. The default value is 1. Note: No effect for IE 6 - 8</dd>
     *              <dt>offset</dt><dd>Number between 0 and 1 indicating where the color stop is positioned.</dd> 
     *          </dl>
     *      </dd>
     *      <p>Linear gradients also have the following property:</p>
     *      <dt>rotation</dt><dd>Linear gradients flow left to right by default. The rotation property allows you to change the flow by rotation. (e.g. A rotation of 180 would make the gradient pain from right to left.)</dd>
     *      <p>Radial gradients have the following additional properties:</p>
     *      <dt>r</dt><dd>Radius of the gradient circle.</dd>
     *      <dt>fx</dt><dd>Focal point x-coordinate of the gradient.</dd>
     *      <dt>fy</dt><dd>Focal point y-coordinate of the gradient.</dd>
     *      <dt>cx</dt><dd>
     *          <p>The x-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>
     *          <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and `VMLShape` classes which are used on Android or IE 6 - 8.</p>
     *      </dd>
     *      <dt>cy</dt><dd>
     *          <p>The y-coordinate of the center of the gradient circle. Determines where the color stop begins. The default value 0.5.</p>
     *          <p><strong>Note: </strong>Currently, this property is not implemented for corresponding `CanvasShape` and `VMLShape` classes which are used on Android or IE 6 - 8.</p>
     *      </dd>
     *  </dl>
         *
         * @config fill
         * @type Object 
         */
        fill: {
                valueFn: "_getDefaultFill",
                
                setter: function(val)
                {
                        var fill,
                                tmpl = this.get("fill") || this._getDefaultFill();
                        fill = (val) ? Y.merge(tmpl, val) : null;
                        if(fill && fill.color)
                        {
                                if(fill.color === undefined || fill.color == "none")
                                {
                                        fill.color = null;
                                }
                        }
                        return fill;
                }
        },

        /**
         * Contains information about the stroke of the shape.
     *  <dl>
     *      <dt>color</dt><dd>The color of the stroke.</dd>
     *      <dt>weight</dt><dd>Number that indicates the width of the stroke.</dd>
     *      <dt>opacity</dt><dd>Number between 0 and 1 that indicates the opacity of the stroke. The default value is 1.</dd>
     *      <dt>dashstyle</dt>Indicates whether to draw a dashed stroke. When set to "none", a solid stroke is drawn. When set to an array, the first index indicates the
     *  length of the dash. The second index indicates the length of gap.
     *      <dt>linecap</dt><dd>Specifies the linecap for the stroke. The following values can be specified:
     *          <dl>
     *              <dt>butt (default)</dt><dd>Specifies a butt linecap.</dd>
     *              <dt>square</dt><dd>Specifies a sqare linecap.</dd>
     *              <dt>round</dt><dd>Specifies a round linecap.</dd>
     *          </dl>
     *      </dd>
     *      <dt>linejoin</dt><dd>Specifies a linejoin for the stroke. The following values can be specified:
     *          <dl>
     *              <dt>round (default)</dt><dd>Specifies that the linejoin will be round.</dd>
     *              <dt>bevel</dt><dd>Specifies a bevel for the linejoin.</dd>
     *              <dt>miter limit</dt><dd>An integer specifying the miter limit of a miter linejoin. If you want to specify a linejoin of miter, you simply specify the limit as opposed to having
     *  separate miter and miter limit values.</dd>
     *          </dl>
     *      </dd>
     *  </dl>
         *
         * @config stroke
         * @type Object
         */
        stroke: {
                valueFn: "_getDefaultStroke",

                setter: function(val)
                {
                        var tmpl = this.get("stroke") || this._getDefaultStroke(),
                wt;
            if(val && val.hasOwnProperty("weight"))
            {
                wt = parseInt(val.weight, 10);
                if(!isNaN(wt))
                {
                    val.weight = wt;
                }
            }
            return (val) ? Y.merge(tmpl, val) : null;
                }
        },
        
        //Not used. Remove in future.
    autoSize: {
                value: false
        },

        // Only implemented in SVG
        // Determines whether the instance will receive mouse events.
        // 
        // @config pointerEvents
        // @type string
        //
        pointerEvents: {
                valueFn: function() 
                {
                        var val = "visiblePainted",
                                node = this.node;
                        if(node)
                        {
                                node.setAttribute("pointer-events", val);
                        }
                        return val;
                },

                setter: function(val)
                {
                        var node = this.node;
                        if(node)
                        {
                                node.setAttribute("pointer-events", val);
                        }
                        return val;
                }
        },

        /**
         * The node used for gradient fills.
         *
         * @config gradientNode
         * @type HTMLElement
     * @private
         */
        gradientNode: {
                setter: function(val)
                {
                        if(Y_LANG.isString(val))
                        {
                                val = this._graphic.getGradientNode("linear", val);
                        }
                        return val;
                }
        },

        //Not used. Remove in future.
    autoDraw: {
                getter: function()
                {
                        return this._graphic.autoDraw;
                }
        },

        /**
         * Dom node for the shape.
         *
         * @config node
         * @type HTMLElement
         * @readOnly
         */
        node: {
                readOnly: true,

                getter: function()
                {
                        return this.node;
                }
        },

        /**
         * Reference to the parent graphic instance
         *
         * @config graphic
         * @type SVGGraphic
         * @readOnly
         */
        graphic: {
                readOnly: true,

                getter: function()
                {
                        return this._graphic;
                }
        }
};
Y.SVGShape = SVGShape;

/**
 * <a href="http://www.w3.org/TR/SVG/">SVG</a> implementation of the <a href="Path.html">`Path`</a> class. 
 * `SVGPath` is not intended to be used directly. Instead, use the <a href="Path.html">`Path`</a> class. 
 * If the browser has <a href="http://www.w3.org/TR/SVG/">SVG</a> capabilities, the <a href="Path.html">`Path`</a> 
 * class will point to the `SVGPath` class.
 *
 * @module graphics
 * @class SVGPath
 * @extends SVGShape
 * @constructor
 */
SVGPath = function(cfg)
{
        SVGPath.superclass.constructor.apply(this, arguments);
};
SVGPath.NAME = "svgPath";
Y.extend(SVGPath, Y.SVGShape, {
    /**
     * Left edge of the path
     *
     * @property _left
     * @type Number
     * @private
     */
    _left: 0,

    /**
     * Right edge of the path
     *
     * @property _right
     * @type Number
     * @private
     */
    _right: 0,
    
    /**
     * Top edge of the path
     *
     * @property _top
     * @type Number
     * @private
     */
    _top: 0, 
    
    /**
     * Bottom edge of the path
     *
     * @property _bottom
     * @type Number
     * @private
     */
    _bottom: 0,

    /**
     * Indicates the type of shape
     *
     * @property _type
     * @readOnly
     * @type String
     * @private
     */
    _type: "path",

    /**
     * Storage for path
     *
     * @property _path
     * @type String
     * @private
     */
        _path: ""
});

SVGPath.ATTRS = Y.merge(Y.SVGShape.ATTRS, {
        /**
         * Indicates the path used for the node.
         *
         * @config path
         * @type String
     * @readOnly
         */
        path: {
                readOnly: true,

                getter: function()
                {
                        return this._path;
                }
        },

        /**
         * Indicates the width of the shape
         * 
         * @config width
         * @type Number
         */
        width: {
                getter: function()
                {
                        var val = Math.max(this._right - this._left, 0);
                        return val;
                }
        },

        /**
         * Indicates the height of the shape
         * 
         * @config height
         * @type Number
         */
        height: {
                getter: function()
                {
                        return Math.max(this._bottom - this._top, 0);
                }
        }
});
Y.SVGPath = SVGPath;
/**
 * <a href="http://www.w3.org/TR/SVG/">SVG</a> implementation of the <a href="Rect.html">`Rect`</a> class. 
 * `SVGRect` is not intended to be used directly. Instead, use the <a href="Rect.html">`Rect`</a> class. 
 * If the browser has <a href="http://www.w3.org/TR/SVG/">SVG</a> capabilities, the <a href="Rect.html">`Rect`</a> 
 * class will point to the `SVGRect` class.
 *
 * @module graphics
 * @class SVGRect
 * @constructor
 */
SVGRect = function()
{
        SVGRect.superclass.constructor.apply(this, arguments);
};
SVGRect.NAME = "svgRect";
Y.extend(SVGRect, Y.SVGShape, {
    /**
     * Indicates the type of shape
     *
     * @property _type
     * @type String
     * @private
     */
    _type: "rect"
 });
SVGRect.ATTRS = Y.SVGShape.ATTRS;
Y.SVGRect = SVGRect;
/**
 * <a href="http://www.w3.org/TR/SVG/">SVG</a> implementation of the <a href="Ellipse.html">`Ellipse`</a> class. 
 * `SVGEllipse` is not intended to be used directly. Instead, use the <a href="Ellipse.html">`Ellipse`</a> class. 
 * If the browser has <a href="http://www.w3.org/TR/SVG/">SVG</a> capabilities, the <a href="Ellipse.html">`Ellipse`</a> 
 * class will point to the `SVGEllipse` class.
 *
 * @module graphics
 * @class SVGEllipse
 * @constructor
 */
SVGEllipse = function(cfg)
{
        SVGEllipse.superclass.constructor.apply(this, arguments);
};

SVGEllipse.NAME = "svgEllipse";

Y.extend(SVGEllipse, SVGShape, {
        /**
         * Indicates the type of shape
         *
         * @property _type
         * @type String
     * @private
         */
        _type: "ellipse",

        /**
         * Updates the shape.
         *
         * @method _draw
         * @private
         */
        _draw: function()
        {
                var node = this.node,
                        w = this.get("width"),
                        h = this.get("height"),
                        x = this.get("x"),
                        y = this.get("y"),
                        xRadius = w * 0.5,
                        yRadius = h * 0.5,
                        cx = x + xRadius,
                        cy = y + yRadius;
                node.setAttribute("rx", xRadius);
                node.setAttribute("ry", yRadius);
                node.setAttribute("cx", cx);
                node.setAttribute("cy", cy);
                this._fillChangeHandler();
                this._strokeChangeHandler();
                this._updateTransform();
        }
});

SVGEllipse.ATTRS = Y.merge(SVGShape.ATTRS, {
        //
        // Horizontal radius for the ellipse. This attribute is not implemented in Canvas.
    // Will add in 3.4.1.
        //
        // @config xRadius
        // @type Number
        // @readOnly
        //
        xRadius: {
                setter: function(val)
                {
                        this.set("width", val/2);
                },

                getter: function()
                {
                        var val = this.get("width");
                        if(val) 
                        {
                                val *= 0.5;
                        }
                        return val;
                }
        },

        //
        // Vertical radius for the ellipse. This attribute is not implemented in Canvas. 
    // Will add in 3.4.1.
        //
        // @config yRadius
        // @type Number
        // @readOnly
        //
        yRadius: {
                setter: function(val)
                {
                        this.set("height", val/2);
                },

                getter: function()
                {
                        var val = this.get("height");
                        if(val) 
                        {
                                val *= 0.5;
                        }
                        return val;
                }
        }
});
Y.SVGEllipse = SVGEllipse;
/**
 * <a href="http://www.w3.org/TR/SVG/">SVG</a> implementation of the <a href="Circle.html">`Circle`</a> class. 
 * `SVGCircle` is not intended to be used directly. Instead, use the <a href="Circle.html">`Circle`</a> class. 
 * If the browser has <a href="http://www.w3.org/TR/SVG/">SVG</a> capabilities, the <a href="Circle.html">`Circle`</a> 
 * class will point to the `SVGCircle` class.
 *
 * @module graphics
 * @class SVGCircle
 * @constructor
 */
 SVGCircle = function(cfg)
 {
    SVGCircle.superclass.constructor.apply(this, arguments);
 };
    
 SVGCircle.NAME = "svgCircle";

 Y.extend(SVGCircle, Y.SVGShape, {    
    
    /**
     * Indicates the type of shape
     *
     * @property _type
     * @type String
     * @private
     */
    _type: "circle",

    /**
     * Updates the shape.
     *
     * @method _draw
     * @private
     */
    _draw: function()
    {
        var node = this.node,
            x = this.get("x"),
            y = this.get("y"),
            radius = this.get("radius"),
            cx = x + radius,
            cy = y + radius;
        node.setAttribute("r", radius);
        node.setAttribute("cx", cx);
        node.setAttribute("cy", cy);
        this._fillChangeHandler();
        this._strokeChangeHandler();
        this._updateTransform();
    }
 });
    
SVGCircle.ATTRS = Y.merge(Y.SVGShape.ATTRS, {
        /**
         * Indicates the width of the shape
         *
         * @config width
         * @type Number
         */
    width: {
        setter: function(val)
        {
            this.set("radius", val/2);
            return val;
        },

        getter: function()
        {
            return this.get("radius") * 2;
        }
    },

        /**
         * Indicates the height of the shape
         *
         * @config height
         * @type Number
         */
    height: {
        setter: function(val)
        {
            this.set("radius", val/2);
            return val;
        },

        getter: function()
        {
            return this.get("radius") * 2;
        }
    },

    /**
     * Radius of the circle
     *
     * @config radius
     * @type Number
     */
    radius: {
        value: 0
    }
});
Y.SVGCircle = SVGCircle;
/**
 * Draws pie slices
 *
 * @module graphics
 * @class SVGPieSlice
 * @constructor
 */
SVGPieSlice = function()
{
        SVGPieSlice.superclass.constructor.apply(this, arguments);
};
SVGPieSlice.NAME = "svgPieSlice";
Y.extend(SVGPieSlice, Y.SVGShape, Y.mix({
    /**
     * Indicates the type of shape
     *
     * @property _type
     * @type String
     * @private
     */
    _type: "path",

        /**
         * Change event listener
         *
         * @private
         * @method _updateHandler
         */
        _draw: function(e)
        {
        var x = this.get("cx"),
            y = this.get("cy"),
            startAngle = this.get("startAngle"),
            arc = this.get("arc"),
            radius = this.get("radius");
        this.clear();
        this.drawWedge(x, y, startAngle, arc, radius);
                this.end();
        }
 }, Y.SVGDrawing.prototype));
SVGPieSlice.ATTRS = Y.mix({
    cx: {
        value: 0
    },

    cy: {
        value: 0
    },
    /**
     * Starting angle in relation to a circle in which to begin the pie slice drawing.
     *
     * @config startAngle
     * @type Number
     */
    startAngle: {
        value: 0
    },

    /**
     * Arc of the slice.
     *
     * @config arc
     * @type Number
     */
    arc: {
        value: 0
    },

    /**
     * Radius of the circle in which the pie slice is drawn
     *
     * @config radius
     * @type Number
     */
    radius: {
        value: 0
    }
}, Y.SVGShape.ATTRS);
Y.SVGPieSlice = SVGPieSlice;
/**
 * <a href="http://www.w3.org/TR/SVG/">SVG</a> implementation of the <a href="Graphic.html">`Graphic`</a> class. 
 * `SVGGraphic` is not intended to be used directly. Instead, use the <a href="Graphic.html">`Graphic`</a> class. 
 * If the browser has <a href="http://www.w3.org/TR/SVG/">SVG</a> capabilities, the <a href="Graphic.html">`Graphic`</a> 
 * class will point to the `SVGGraphic` class.
 *
 * @module graphics
 * @class SVGGraphic
 * @constructor
 */
SVGGraphic = function(cfg) {
    SVGGraphic.superclass.constructor.apply(this, arguments);
};

SVGGraphic.NAME = "svgGraphic";

SVGGraphic.ATTRS = {
    /**
     * Whether or not to render the `Graphic` automatically after to a specified parent node after init. This can be a Node instance or a CSS selector string.
     * 
     * @config render
     * @type Node | String 
     */
    render: {},
        
    /**
         * Unique id for class instance.
         *
         * @config id
         * @type String
         */
        id: {
                valueFn: function()
                {
                        return Y.guid();
                },

                setter: function(val)
                {
                        var node = this._node;
                        if(node)
                        {
                                node.setAttribute("id", val);
                        }
                        return val;
                }
        },

    /**
     * Key value pairs in which a shape instance is associated with its id.
     *
     *  @config shapes
     *  @type Object
     *  @readOnly
     */
    shapes: {
        readOnly: true,

        getter: function()
        {
            return this._shapes;
        }
    },

    /**
     *  Object containing size and coordinate data for the content of a Graphic in relation to the coordSpace node.
     *
     *  @config contentBounds
     *  @type Object 
     *  @readOnly
     */
    contentBounds: {
        readOnly: true,

        getter: function()
        {
            return this._contentBounds;
        }
    },

    /**
     *  The html element that represents to coordinate system of the Graphic instance.
     *
     *  @config node
     *  @type HTMLElement
     *  @readOnly
     */
    node: {
        readOnly: true,

        getter: function()
        {
            return this._node;
        }
    },
    
        /**
         * Indicates the width of the `Graphic`. 
         *
         * @config width
         * @type Number
         */
    width: {
        setter: function(val)
        {
            if(this._node)
            {
                this._node.style.width = val + "px";
            }
            return val; 
        }
    },

        /**
         * Indicates the height of the `Graphic`. 
         *
         * @config height 
         * @type Number
         */
    height: {
        setter: function(val)
        {
            if(this._node)
            {
                this._node.style.height = val  + "px";
            }
            return val;
        }
    },

    /**
     *  Determines how the size of instance is calculated. If true, the width and height are determined by the size of the contents.
     *  If false, the width and height values are either explicitly set or determined by the size of the parent node's dimensions.
     *
     *  @config autoSize
     *  @type Boolean
     *  @default false
     */
    autoSize: {
        value: false
    },

    /**
     * The contentBounds will resize to greater values but not to smaller values. (for performance)
     * When resizing the contentBounds down is desirable, set the resizeDown value to true.
     *
     * @config resizeDown 
     * @type Boolean
     */
    resizeDown: {
        getter: function()
        {
            return this._resizeDown;
        },

        setter: function(val)
        {
            this._resizeDown = val;
            this._redraw();
            return val;
        }
    },

        /**
         * Indicates the x-coordinate for the instance.
         *
         * @config x
         * @type Number
         */
    x: {
        getter: function()
        {
            return this._x;
        },

        setter: function(val)
        {
            this._x = val;
            if(this._node)
            {
                this._node.style.left = val + "px";
            }
            return val;
        }
    },

        /**
         * Indicates the y-coordinate for the instance.
         *
         * @config y
         * @type Number
         */
    y: {
        getter: function()
        {
            return this._y;
        },

        setter: function(val)
        {
            this._y = val;
            if(this._node)
            {
                this._node.style.top = val + "px";
            }
            return val;
        }
    },

    /**
     * Indicates whether or not the instance will automatically redraw after a change is made to a shape.
     * This property will get set to false when batching operations.
     *
     * @config autoDraw
     * @type Boolean
     * @default true
     * @private
     */
    autoDraw: {
        value: true
    },
    
    visible: {
        value: true,

        setter: function(val)
        {
            this._toggleVisible(val);
            return val;
        }
    },

    //
    //  Indicates the pointer-events setting for the svg:svg element.
    //
    //  @config pointerEvents
    //  @type String
    //
    pointerEvents: {
        value: "none"
    }
};

Y.extend(SVGGraphic, Y.BaseGraphic, {
    /**
     * Storage for `x` attribute.
     *
     * @property _x
     * @type Number
     * @private
     */
    _x: 0,

    /**
     * Storage for `y` attribute.
     *
     * @property _y
     * @type Number
     * @private
     */
    _y: 0,

    /**
     * Gets the current position of the graphic instance in page coordinates.
     *
     * @method getXY
     * @return Array The XY position of the shape.
     */
    getXY: function()
    {
        var node = Y.one(this._node),
            xy;
        if(node)
        {
            xy = node.getXY();
        }
        return xy;
    },

    /**
     * @private
     * @property _resizeDown 
     * @type Boolean
     */
    _resizeDown: false,

    /**
     * Initializes the class.
     *
     * @method initializer
     * @private
     */
    initializer: function() {
        var render = this.get("render");
        this._shapes = {};
                this._contentBounds = {
            left: 0,
            top: 0,
            right: 0,
            bottom: 0
        };
        this._gradients = {};
        this._node = DOCUMENT.createElement('div');
        this._node.style.position = "absolute";
        this._node.style.left = this.get("x") + "px";
        this._node.style.top = this.get("y") + "px";
        this._contentNode = this._createGraphics();
        this._contentNode.setAttribute("id", this.get("id"));
        this._node.appendChild(this._contentNode);
        if(render)
        {
            this.render(render);
        }
    },

    /**
     * Adds the graphics node to the dom.
     * 
     * @method render
     * @param {HTMLElement} parentNode node in which to render the graphics node into.
     */
    render: function(render) {
        var parentNode = Y.one(render),
            w = this.get("width") || parseInt(parentNode.getComputedStyle("width"), 10),
            h = this.get("height") || parseInt(parentNode.getComputedStyle("height"), 10);
        parentNode = parentNode || DOCUMENT.body;
        parentNode.appendChild(this._node);
        this.parentNode = parentNode;
        this.set("width", w);
        this.set("height", h);
        this.parentNode = parentNode;
        return this;
    },

    /**
     * Removes all nodes.
     *
     * @method destroy
     */
    destroy: function()
    {
        this.removeAllShapes();
        this._removeChildren(this._node);
        if(this._node && this._node.parentNode)
        {
            this._node.parentNode.removeChild(this._node);
        }
    },

    /**
     * Generates a shape instance by type.
     *
     * @method addShape
     * @param {Object} cfg attributes for the shape
     * @return Shape
     */
    addShape: function(cfg)
    {
        cfg.graphic = this;
        var shapeClass = this._getShapeClass(cfg.type),
            shape = new shapeClass(cfg);
        this._appendShape(shape);
        return shape;
    },

    /**
     * Adds a shape instance to the graphic instance.
     *
     * @method _appendShape
     * @param {Shape} shape The shape instance to be added to the graphic.
     * @private
     */
    _appendShape: function(shape)
    {
        var node = shape.node,
            parentNode = this._frag || this._contentNode;
        if(this.get("autoDraw")) 
        {
            parentNode.appendChild(node);
        }
        else
        {
            this._getDocFrag().appendChild(node);
        }
    },

    /**
     * Removes a shape instance from from the graphic instance.
     *
     * @method removeShape
     * @param {Shape|String} shape The instance or id of the shape to be removed.
     */
    removeShape: function(shape)
    {
        if(!(shape instanceof SVGShape))
        {
            if(Y_LANG.isString(shape))
            {
                shape = this._shapes[shape];
            }
        }
        if(shape && shape instanceof SVGShape)
        {
            shape.destroy();
            delete this._shapes[shape.get("id")];
        }
        if(this.get("autoDraw")) 
        {
            this._redraw();
        }
        return shape;
    },

    /**
     * Removes all shape instances from the dom.
     *
     * @method removeAllShapes
     */
    removeAllShapes: function()
    {
        var shapes = this._shapes,
            i;
        for(i in shapes)
        {
            if(shapes.hasOwnProperty(i))
            {
                shapes[i].destroy();
            }
        }
        this._shapes = {};
    },
    
    /**
     * Removes all child nodes.
     *
     * @method _removeChildren
     * @param {HTMLElement} node
     * @private
     */
    _removeChildren: function(node)
    {
        if(node.hasChildNodes())
        {
            var child;
            while(node.firstChild)
            {
                child = node.firstChild;
                this._removeChildren(child);
                node.removeChild(child);
            }
        }
    },

    /**
     * Clears the graphics object.
     *
     * @method clear
     */
    clear: function() {
        this.removeAllShapes();
    },

    /**
     * Toggles visibility
     *
     * @method _toggleVisible
     * @param {Boolean} val indicates visibilitye
     * @private
     */
    _toggleVisible: function(val)
    {
        var i,
            shapes = this._shapes,
            visibility = val ? "visible" : "hidden";
        if(shapes)
        {
            for(i in shapes)
            {
                if(shapes.hasOwnProperty(i))
                {
                    shapes[i].set("visible", val);
                }
            }
        }
        this._contentNode.style.visibility = visibility;
        this._node.style.visibility = visibility;
    },

    /**
     * Returns a shape class. Used by `addShape`. 
     *
     * @method _getShapeClass
     * @param {Shape | String} val Indicates which shape class. 
     * @return Function 
     * @private
     */
    _getShapeClass: function(val)
    {
        var shape = this._shapeClass[val];
        if(shape)
        {
            return shape;
        }
        return val;
    },

    /**
     * Look up for shape classes. Used by `addShape` to retrieve a class for instantiation.
     *
     * @property _shapeClass
     * @type Object
     * @private
     */
    _shapeClass: {
        circle: Y.SVGCircle,
        rect: Y.SVGRect,
        path: Y.SVGPath,
        ellipse: Y.SVGEllipse,
        pieslice: Y.SVGPieSlice
    },
    
    /**
     * Returns a shape based on the id of its dom node.
     *
     * @method getShapeById
     * @param {String} id Dom id of the shape's node attribute.
     * @return Shape
     */
    getShapeById: function(id)
    {
        var shape = this._shapes[id];
        return shape;
    },

        /**
         * Allows for creating multiple shapes in order to batch appending and redraw operations.
         *
         * @method batch
         * @param {Function} method Method to execute.
         */
    batch: function(method)
    {
        var autoDraw = this.get("autoDraw");
        this.set("autoDraw", false);
        method();
        this._redraw();
        this.set("autoDraw", autoDraw);
    },
    
    /**
     * Returns a document fragment to for attaching shapes.
     *
     * @method _getDocFrag
     * @return DocumentFragment
     * @private
     */
    _getDocFrag: function()
    {
        if(!this._frag)
        {
            this._frag = DOCUMENT.createDocumentFragment();
        }
        return this._frag;
    },

    /**
     * Redraws all shapes.
     *
     * @method _redraw
     * @private
     */
    _redraw: function()
    {
        var box = this.get("resizeDown") ? this._getUpdatedContentBounds() : this._contentBounds;
        this._contentNode.style.left = box.left + "px";
        this._contentNode.style.top = box.top + "px";
        this._contentNode.setAttribute("width", box.width);
        this._contentNode.setAttribute("height", box.height);
        this._contentNode.style.width = box.width + "px";
        this._contentNode.style.height = box.height + "px";
        this._contentNode.setAttribute("viewBox", "" + box.left + " " + box.top + " " + box.width + " " + box.height + "");
        if(this.get("autoSize"))
        {
            this.set("width", box.right);
            this.set("height", box.bottom);
        }
        if(this._frag)
        {
            this._contentNode.appendChild(this._frag);
            this._frag = null;
        }
    },

    /**
     * Adds a shape to the redraw queue and calculates the contentBounds. Used internally 
     * by `Shape` instances.
     *
     * @method addToRedrawQueue
     * @param shape {SVGShape}
     * @protected
     */
    addToRedrawQueue: function(shape)
    {
        var shapeBox,
            box;
        this._shapes[shape.get("id")] = shape;
        if(!this.get("resizeDown"))
        {
            shapeBox = shape.getBounds();
            box = this._contentBounds;
            box.left = box.left < shapeBox.left ? box.left : shapeBox.left;
            box.top = box.top < shapeBox.top ? box.top : shapeBox.top;
            box.right = box.right > shapeBox.right ? box.right : shapeBox.right;
            box.bottom = box.bottom > shapeBox.bottom ? box.bottom : shapeBox.bottom;
            box.width = box.right - box.left;
            box.height = box.bottom - box.top;
            this._contentBounds = box;
        }
        if(this.get("autoDraw")) 
        {
            this._redraw();
        }
    },
    
    /**
     * Recalculates and returns the `contentBounds` for the `Graphic` instance.
     *
     * @method _getUpdatedContentBounds
     * @return {Object} 
     * @private
     */
    _getUpdatedContentBounds: function()
    {
        var bounds,
            i,
            shape,
            queue = this._shapes,
            box = {
                left: 0,
                top: 0,
                right: 0,
                bottom: 0
            };
        for(i in queue)
        {
            if(queue.hasOwnProperty(i))
            {
                shape = queue[i];
                bounds = shape.getBounds();
                box.left = Math.min(box.left, bounds.left);
                box.top = Math.min(box.top, bounds.top);
                box.right = Math.max(box.right, bounds.right);
                box.bottom = Math.max(box.bottom, bounds.bottom);
            }
        }
        box.width = box.right - box.left;
        box.height = box.bottom - box.top;
        this._contentBounds = box;
        return box;
    },

    /**
     * Creates a contentNode element
     *
     * @method _createGraphics
     * @private
     */
    _createGraphics: function() {
        var contentNode = this._createGraphicNode("svg"),
            pointerEvents = this.get("pointerEvents");
        contentNode.style.position = "absolute";
        contentNode.style.top = "px";
        contentNode.style.left = "0px";
        contentNode.style.overflow = "auto";
        contentNode.setAttribute("overflow", "auto");
        contentNode.setAttribute("pointer-events", pointerEvents);
        return contentNode;
    },

    /**
     * Creates a graphic node
     *
     * @method _createGraphicNode
     * @param {String} type node type to create
     * @param {String} pe specified pointer-events value
     * @return HTMLElement
     * @private
     */
    _createGraphicNode: function(type, pe)
    {
        var node = DOCUMENT.createElementNS("http://www.w3.org/2000/svg", "svg:" + type),
            v = pe || "none";
        if(type !== "defs" && type !== "stop" && type !== "linearGradient" && type != "radialGradient")
        {
            node.setAttribute("pointer-events", v);
        }
        return node;
    },

    /**
     * Returns a reference to a gradient definition based on an id and type.
     *
     * @method getGradientNode
     * @param {String} key id that references the gradient definition
     * @param {String} type description of the gradient type
     * @return HTMLElement
     * @protected
     */
    getGradientNode: function(key, type)
    {
        var gradients = this._gradients,
            gradient,
            nodeType = type + "Gradient";
        if(gradients.hasOwnProperty(key) && gradients[key].tagName.indexOf(type) > -1)
        {
            gradient = this._gradients[key];
        }
        else
        {
            gradient = this._createGraphicNode(nodeType);
            if(!this._defs)
            {
                this._defs = this._createGraphicNode("defs");
                this._contentNode.appendChild(this._defs);
            }
            this._defs.appendChild(gradient);
            key = key || "gradient" + Math.round(100000 * Math.random());
            gradient.setAttribute("id", key);
            if(gradients.hasOwnProperty(key))
            {
                this._defs.removeChild(gradients[key]);
            }
            gradients[key] = gradient;
        }
        return gradient;
    }

});

Y.SVGGraphic = SVGGraphic;



}, '3.4.1' ,{requires:['graphics'], skinnable:false});
